/*
 * Decompiled with CFR 0.152.
 */
package io.lettuce.core.cluster;

import io.lettuce.core.cluster.ClusterClientOptions;
import io.lettuce.core.cluster.ClusterEventListener;
import io.lettuce.core.cluster.ClusterTopologyRefreshOptions;
import io.lettuce.core.cluster.RedisClusterClient;
import io.lettuce.core.event.cluster.AdaptiveRefreshTriggeredEvent;
import io.lettuce.core.resource.ClientResources;
import io.netty.util.concurrent.EventExecutorGroup;
import io.netty.util.internal.logging.InternalLogger;
import io.netty.util.internal.logging.InternalLoggerFactory;
import java.time.Duration;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

class ClusterTopologyRefreshScheduler
implements Runnable,
ClusterEventListener {
    private static final InternalLogger logger = InternalLoggerFactory.getInstance(ClusterTopologyRefreshScheduler.class);
    private static final ClusterTopologyRefreshOptions FALLBACK_OPTIONS = ClusterTopologyRefreshOptions.create();
    private final RedisClusterClient redisClusterClient;
    private final ClientResources clientResources;
    private final ClusterTopologyRefreshTask clusterTopologyRefreshTask;
    private final AtomicReference<Timeout> timeoutRef = new AtomicReference();

    ClusterTopologyRefreshScheduler(RedisClusterClient redisClusterClient, ClientResources clientResources) {
        this.redisClusterClient = redisClusterClient;
        this.clientResources = clientResources;
        this.clusterTopologyRefreshTask = new ClusterTopologyRefreshTask(redisClusterClient);
    }

    @Override
    public void run() {
        logger.debug("ClusterTopologyRefreshScheduler.run()");
        if (this.isEventLoopActive() && this.redisClusterClient.getClusterClientOptions() != null) {
            if (!this.redisClusterClient.getClusterClientOptions().isRefreshClusterView()) {
                logger.debug("Periodic ClusterTopologyRefresh is disabled");
                return;
            }
        } else {
            logger.debug("Periodic ClusterTopologyRefresh is disabled");
            return;
        }
        this.clientResources.eventExecutorGroup().submit((Runnable)this.clusterTopologyRefreshTask);
    }

    private boolean indicateTopologyRefreshSignal() {
        logger.debug("ClusterTopologyRefreshScheduler.indicateTopologyRefreshSignal()");
        if (!this.acquireTimeout()) {
            return false;
        }
        return this.scheduleRefresh();
    }

    private boolean scheduleRefresh() {
        if (this.isEventLoopActive() && this.redisClusterClient.getClusterClientOptions() != null) {
            this.clientResources.eventExecutorGroup().submit((Runnable)this.clusterTopologyRefreshTask);
            return true;
        }
        logger.debug("ClusterTopologyRefresh is disabled");
        return false;
    }

    private boolean isEventLoopActive() {
        EventExecutorGroup eventExecutors = this.clientResources.eventExecutorGroup();
        return !eventExecutors.isShuttingDown() && !eventExecutors.isShutdown() && !eventExecutors.isTerminated();
    }

    private boolean acquireTimeout() {
        Timeout existingTimeout = this.timeoutRef.get();
        if (existingTimeout != null && !existingTimeout.isExpired()) {
            return false;
        }
        ClusterTopologyRefreshOptions refreshOptions = this.getClusterTopologyRefreshOptions();
        Timeout timeout = new Timeout(refreshOptions.getAdaptiveRefreshTimeout());
        return this.timeoutRef.compareAndSet(existingTimeout, timeout);
    }

    @Override
    public void onAskRedirection() {
        if (this.isEnabled(ClusterTopologyRefreshOptions.RefreshTrigger.ASK_REDIRECT)) {
            this.indicateTopologyRefreshSignal();
        }
    }

    @Override
    public void onMovedRedirection() {
        if (this.isEnabled(ClusterTopologyRefreshOptions.RefreshTrigger.MOVED_REDIRECT) && this.indicateTopologyRefreshSignal()) {
            this.emitAdaptiveRefreshScheduledEvent();
        }
    }

    @Override
    public void onReconnectAttempt(int attempt) {
        if (this.isEnabled(ClusterTopologyRefreshOptions.RefreshTrigger.PERSISTENT_RECONNECTS) && attempt >= this.getClusterTopologyRefreshOptions().getRefreshTriggersReconnectAttempts() && this.indicateTopologyRefreshSignal()) {
            this.emitAdaptiveRefreshScheduledEvent();
        }
    }

    @Override
    public void onUncoveredSlot(int slot) {
        if (this.isEnabled(ClusterTopologyRefreshOptions.RefreshTrigger.UNCOVERED_SLOT) && this.indicateTopologyRefreshSignal()) {
            this.emitAdaptiveRefreshScheduledEvent();
        }
    }

    @Override
    public void onUnknownNode() {
        if (this.isEnabled(ClusterTopologyRefreshOptions.RefreshTrigger.UNKNOWN_NODE) && this.indicateTopologyRefreshSignal()) {
            this.emitAdaptiveRefreshScheduledEvent();
        }
    }

    private void emitAdaptiveRefreshScheduledEvent() {
        AdaptiveRefreshTriggeredEvent event = new AdaptiveRefreshTriggeredEvent(this.redisClusterClient::getPartitions, this::scheduleRefresh);
        this.clientResources.eventBus().publish(event);
    }

    private ClusterTopologyRefreshOptions getClusterTopologyRefreshOptions() {
        ClusterClientOptions clusterClientOptions = this.redisClusterClient.getClusterClientOptions();
        if (clusterClientOptions != null) {
            return clusterClientOptions.getTopologyRefreshOptions();
        }
        return FALLBACK_OPTIONS;
    }

    private boolean isEnabled(ClusterTopologyRefreshOptions.RefreshTrigger refreshTrigger) {
        return this.getClusterTopologyRefreshOptions().getAdaptiveRefreshTriggers().contains((Object)refreshTrigger);
    }

    private static class ClusterTopologyRefreshTask
    implements Runnable {
        private final RedisClusterClient redisClusterClient;
        private final AtomicBoolean unique = new AtomicBoolean();

        ClusterTopologyRefreshTask(RedisClusterClient redisClusterClient) {
            this.redisClusterClient = redisClusterClient;
        }

        @Override
        public void run() {
            if (this.unique.compareAndSet(false, true)) {
                try {
                    this.doRun();
                }
                finally {
                    this.unique.set(false);
                }
                return;
            }
            if (logger.isDebugEnabled()) {
                logger.debug("ClusterTopologyRefreshTask already in progress");
            }
        }

        void doRun() {
            if (logger.isDebugEnabled()) {
                logger.debug("ClusterTopologyRefreshTask requesting partitions from {}", this.redisClusterClient.getTopologyRefreshSource());
            }
            try {
                this.redisClusterClient.reloadPartitions();
            }
            catch (Exception e) {
                logger.warn("Cannot refresh Redis Cluster topology", (Throwable)e);
            }
        }
    }

    private class Timeout {
        private final long expiresMs;

        public Timeout(Duration duration) {
            this.expiresMs = System.currentTimeMillis() + duration.toMillis();
        }

        public boolean isExpired() {
            return this.expiresMs < System.currentTimeMillis();
        }

        public long remaining() {
            long diff = this.expiresMs - System.currentTimeMillis();
            if (diff > 0L) {
                return diff;
            }
            return 0L;
        }
    }
}

