/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.connection;

import com.mongodb.MongoException;
import com.mongodb.MongoInternalException;
import com.mongodb.MongoInterruptedException;
import com.mongodb.MongoServerUnavailableException;
import com.mongodb.MongoTimeoutException;
import com.mongodb.annotations.ThreadSafe;
import com.mongodb.assertions.Assertions;
import com.mongodb.internal.connection.ConcurrentLinkedDeque;
import com.mongodb.internal.connection.Pool;
import com.mongodb.lang.Nullable;
import java.util.Iterator;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Consumer;
import java.util.function.Supplier;

public class ConcurrentPool<T>
implements Pool<T> {
    public static final int INFINITE_SIZE = Integer.MAX_VALUE;
    private final int maxSize;
    private final ItemFactory<T> itemFactory;
    private final ConcurrentLinkedDeque<T> available = new ConcurrentLinkedDeque();
    private final StateAndPermits stateAndPermits;
    private final String poolClosedMessage;

    public ConcurrentPool(int maxSize, ItemFactory<T> itemFactory) {
        this(maxSize, itemFactory, "The pool is closed");
    }

    public ConcurrentPool(int maxSize, ItemFactory<T> itemFactory, String poolClosedMessage) {
        Assertions.assertTrue(maxSize > 0);
        this.maxSize = maxSize;
        this.itemFactory = itemFactory;
        this.stateAndPermits = new StateAndPermits(maxSize, this::poolClosedException);
        this.poolClosedMessage = Assertions.notNull("poolClosedMessage", poolClosedMessage);
    }

    @Override
    public void release(T t) {
        this.release(t, false);
    }

    @Override
    public void release(T t, boolean prune) {
        if (t == null) {
            throw new IllegalArgumentException("Can not return a null item to the pool");
        }
        if (this.stateAndPermits.closed()) {
            this.close(t);
            return;
        }
        if (prune) {
            this.close(t);
        } else {
            this.available.addLast(t);
        }
        this.stateAndPermits.releasePermit();
    }

    @Override
    public T get() {
        return this.get(-1L, TimeUnit.MILLISECONDS);
    }

    @Override
    public T get(long timeout, TimeUnit timeUnit) {
        this.stateAndPermits.throwIfClosedOrPaused();
        if (!this.stateAndPermits.acquirePermitFair(timeout, timeUnit)) {
            throw new MongoTimeoutException(String.format("Timeout waiting for a pooled item after %d %s", new Object[]{timeout, timeUnit}));
        }
        T t = this.available.pollLast();
        if (t == null) {
            t = this.createNewAndReleasePermitIfFailure();
        }
        return t;
    }

    @Nullable
    T getImmediateUnfair() {
        this.stateAndPermits.throwIfClosedOrPaused();
        T element = null;
        if (this.stateAndPermits.acquirePermitImmediateUnfair() && (element = (T)this.available.pollLast()) == null) {
            this.stateAndPermits.releasePermit();
        }
        return element;
    }

    public void prune() {
        Object cur;
        Prune shouldPrune;
        Iterator iter = this.available.iterator();
        while (iter.hasNext() && (shouldPrune = this.itemFactory.shouldPrune(cur = iter.next())) != Prune.STOP) {
            boolean removed;
            if (shouldPrune != Prune.YES || !(removed = iter.reportingRemove())) continue;
            this.close(cur);
        }
    }

    public void ensureMinSize(int minSize, Consumer<T> initAndRelease) {
        this.stateAndPermits.throwIfClosedOrPaused();
        while (this.getCount() < minSize && this.stateAndPermits.acquirePermitFair(0L, TimeUnit.MILLISECONDS)) {
            initAndRelease.accept(this.createNewAndReleasePermitIfFailure());
        }
    }

    private T createNewAndReleasePermitIfFailure() {
        try {
            T newMember = this.itemFactory.create();
            if (newMember == null) {
                throw new MongoInternalException("The factory for the pool created a null item");
            }
            return newMember;
        }
        catch (RuntimeException e) {
            this.stateAndPermits.releasePermit();
            throw e;
        }
    }

    boolean acquirePermit(long timeout, TimeUnit timeUnit) {
        return this.stateAndPermits.acquirePermitFair(timeout, timeUnit);
    }

    @Override
    public void close() {
        if (this.stateAndPermits.close()) {
            Iterator iter = this.available.iterator();
            while (iter.hasNext()) {
                Object t = iter.next();
                this.close(t);
                iter.remove();
            }
        }
    }

    int getMaxSize() {
        return this.maxSize;
    }

    public int getInUseCount() {
        return this.maxSize - this.stateAndPermits.permits();
    }

    public int getAvailableCount() {
        return this.available.size();
    }

    public int getCount() {
        return this.getInUseCount() + this.getAvailableCount();
    }

    public String toString() {
        StringBuilder buf = new StringBuilder();
        buf.append("pool: ").append(" maxSize: ").append(ConcurrentPool.sizeToString(this.maxSize)).append(" availableCount ").append(this.getAvailableCount()).append(" inUseCount ").append(this.getInUseCount());
        return buf.toString();
    }

    private void close(T t) {
        try {
            this.itemFactory.close(t);
        }
        catch (RuntimeException runtimeException) {
            // empty catch block
        }
    }

    void ready() {
        this.stateAndPermits.ready();
    }

    void pause(Supplier<MongoException> causeSupplier) {
        this.stateAndPermits.pause(causeSupplier);
    }

    MongoServerUnavailableException poolClosedException() {
        return new MongoServerUnavailableException(this.poolClosedMessage);
    }

    static boolean isPoolClosedException(Throwable e) {
        return e instanceof MongoServerUnavailableException;
    }

    static String sizeToString(int size) {
        return size == Integer.MAX_VALUE ? "infinite" : Integer.toString(size);
    }

    @ThreadSafe
    private static final class StateAndPermits {
        private final Supplier<MongoServerUnavailableException> poolClosedExceptionSupplier;
        private final ReadWriteLock lock;
        private final Condition permitAvailableOrClosedOrPausedCondition;
        private volatile boolean paused;
        private volatile boolean closed;
        private final int maxPermits;
        private volatile int permits;
        @Nullable
        private Supplier<MongoException> causeSupplier;

        StateAndPermits(int maxPermits, Supplier<MongoServerUnavailableException> poolClosedExceptionSupplier) {
            this.poolClosedExceptionSupplier = poolClosedExceptionSupplier;
            this.lock = new ReentrantReadWriteLock(true);
            this.permitAvailableOrClosedOrPausedCondition = this.lock.writeLock().newCondition();
            this.paused = false;
            this.closed = false;
            this.maxPermits = maxPermits;
            this.permits = maxPermits;
            this.causeSupplier = null;
        }

        int permits() {
            return this.permits;
        }

        boolean acquirePermitImmediateUnfair() {
            if (!this.lock.writeLock().tryLock()) {
                this.lock.writeLock().lock();
            }
            try {
                this.throwIfClosedOrPaused();
                if (this.permits > 0) {
                    --this.permits;
                    boolean bl = true;
                    return bl;
                }
                boolean bl = false;
                return bl;
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        boolean acquirePermitFair(long timeout, TimeUnit unit) throws MongoInterruptedException {
            long remainingNanos = unit.toNanos(timeout);
            try {
                this.lock.writeLock().lockInterruptibly();
            }
            catch (InterruptedException e) {
                throw new MongoInterruptedException(null, e);
            }
            try {
                while (this.permits == 0) {
                    this.throwIfClosedOrPaused();
                    try {
                        if (timeout < 0L || remainingNanos == Long.MAX_VALUE) {
                            this.permitAvailableOrClosedOrPausedCondition.await();
                            continue;
                        }
                        if (remainingNanos < 0L) {
                            boolean e = false;
                            return e;
                        }
                        remainingNanos = this.permitAvailableOrClosedOrPausedCondition.awaitNanos(remainingNanos);
                    }
                    catch (InterruptedException e) {
                        throw new MongoInterruptedException(null, e);
                    }
                }
                Assertions.assertTrue(this.permits > 0);
                this.throwIfClosedOrPaused();
                --this.permits;
                boolean bl = true;
                return bl;
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        void releasePermit() {
            this.lock.writeLock().lock();
            try {
                Assertions.assertTrue(this.permits < this.maxPermits);
                ++this.permits;
                this.permitAvailableOrClosedOrPausedCondition.signal();
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        void pause(Supplier<MongoException> causeSupplier) {
            this.lock.writeLock().lock();
            try {
                if (!this.paused) {
                    this.paused = true;
                    this.permitAvailableOrClosedOrPausedCondition.signalAll();
                }
                this.causeSupplier = Assertions.assertNotNull(causeSupplier);
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        void ready() {
            if (this.paused) {
                this.lock.writeLock().lock();
                try {
                    this.paused = false;
                    this.causeSupplier = null;
                }
                finally {
                    this.lock.writeLock().unlock();
                }
            }
        }

        boolean close() {
            if (!this.closed) {
                this.lock.writeLock().lock();
                try {
                    if (!this.closed) {
                        this.closed = true;
                        this.permitAvailableOrClosedOrPausedCondition.signalAll();
                        boolean bl = true;
                        return bl;
                    }
                }
                finally {
                    this.lock.writeLock().unlock();
                }
            }
            return false;
        }

        void throwIfClosedOrPaused() {
            if (this.closed) {
                throw this.poolClosedExceptionSupplier.get();
            }
            if (this.paused) {
                this.lock.readLock().lock();
                try {
                    if (this.paused) {
                        throw Assertions.assertNotNull(Assertions.assertNotNull(this.causeSupplier).get());
                    }
                }
                finally {
                    this.lock.readLock().unlock();
                }
            }
        }

        boolean closed() {
            return this.closed;
        }
    }

    public static interface ItemFactory<T> {
        public T create();

        public void close(T var1);

        public Prune shouldPrune(T var1);
    }

    public static enum Prune {
        YES,
        NO,
        STOP;

    }
}

